const { get, run, query } = require("../config/db");
const bcrypt = require("bcryptjs");

const User = {
  // Helper to convert user object to safe format with string IDs
  toSafeObject: (user) => {
    if (!user) return null;
    return {
      id: user.id.toString(),
      name: user.name,
      email: user.email,
      avatar: user.avatar,
      role: user.role,
      created_at: user.created_at,
      updated_at: user.updated_at,
    };
  },

  // Create a new user
  create: async (userData) => {
    const {
      name,
      email,
      password,
      avatar,
      role = "customer",
      fcmTokens = "[]",
    } = userData;

    // Hash password
    const salt = await bcrypt.genSalt(10);
    const hashedPassword = await bcrypt.hash(password, salt);

    const result = run(
      `INSERT INTO users (name, email, password, avatar, role, fcmTokens)
       VALUES (?, ?, ?, ?, ?, ?)`,
      [name, email.toLowerCase(), hashedPassword, avatar, role, fcmTokens],
    );

    const user = User.findById(result.lastInsertRowid);
    return User.toSafeObject(user);
  },

  // Find user by ID
  findById: (id) => {
    return get("SELECT * FROM users WHERE id = ?", [id]);
  },

  // Find user by email (case insensitive)
  findByEmail: (email) => {
    return get("SELECT * FROM users WHERE LOWER(email) = LOWER(?)", [email]);
  },

  // Find all users
  findAll: (options = {}) => {
    let sql = "SELECT * FROM users";
    const params = [];

    if (options.where) {
      const conditions = [];
      if (options.where.role) {
        conditions.push("role = ?");
        params.push(options.where.role);
      }
      if (conditions.length > 0) {
        sql += " WHERE " + conditions.join(" AND ");
      }
    }

    if (options.order) {
      sql += " ORDER BY " + options.order;
    }

    const users = query(sql, params);
    return users.map((user) => User.toSafeObject(user));
  },

  // Update user
  update: (id, updates) => {
    const fields = [];
    const values = [];

    Object.entries(updates).forEach(([key, value]) => {
      if (value !== undefined) {
        fields.push(`${key} = ?`);
        values.push(value);
      }
    });

    if (fields.length === 0) return null;

    values.push(id);
    run(`UPDATE users SET ${fields.join(", ")} WHERE id = ?`, values);

    const user = User.findById(id);
    return User.toSafeObject(user);
  },

  // Delete user
  delete: (id) => {
    const user = User.findById(id);
    run("DELETE FROM users WHERE id = ?", [id]);
    return User.toSafeObject(user);
  },

  // Compare password
  comparePassword: async (user, candidatePassword) => {
    return await bcrypt.compare(candidatePassword, user.password);
  },

  // Get teachers only
  getTeachers: () => {
    const teachers = query(
      "SELECT id, name, email, avatar FROM users WHERE role = ?",
      ["teacher"],
    );
    return teachers.map((teacher) => ({
      ...teacher,
      id: teacher.id.toString(),
    }));
  },

  // Update FCM tokens
  addFcmToken: (id, token) => {
    const user = User.findById(id);
    if (!user) return null;

    let tokens = JSON.parse(user.fcmTokens || "[]");
    if (!tokens.includes(token)) {
      tokens.push(token);
      run("UPDATE users SET fcmTokens = ? WHERE id = ?", [
        JSON.stringify(tokens),
        id,
      ]);
    }

    const updatedUser = User.findById(id);
    return User.toSafeObject(updatedUser);
  },
};

module.exports = User;
