const Student = require("../models/Student");
const sharp = require("sharp");
const fs = require("fs");
const path = require("path");

// Create Student
exports.createStudent = async (req, res) => {
  try {
    if (!req.file) {
      return res.status(400).json({ error: "Photo is required" });
    }
    const { name, age, phone, fatherName, description, teacherId } = req.body;

    // Generate filename
    const timestamp = Date.now();
    const cleanFilename = `student-${timestamp}.webp`;
    const webpPath = path.join(
      __dirname,
      "..",
      "uploads",
      "webp",
      cleanFilename,
    );

    // Ensure webp directory exists
    const webpDir = path.join(__dirname, "..", "uploads", "webp");
    if (!fs.existsSync(webpDir)) {
      fs.mkdirSync(webpDir, { recursive: true });
    }

    // Process image
    await sharp(req.file.path)
      .resize(500, 500)
      .webp({ quality: 80 })
      .toFile(webpPath);

    // Create student
    const student = Student.create({
      name,
      age: parseInt(age),
      phone,
      fatherName,
      description,
      teacherId,
      photo: req.file.originalname,
      photoWebP: cleanFilename,
    });

    // Clean temp file
    fs.unlinkSync(req.file.path);

    res.status(201).json({
      message: "Student created",
      student: {
        id: student.id,
        name: student.name,
        phone: student.phone,
        age: student.age,
        fatherName: student.fatherName,
        description: student.description,
        teacherId: student.teacherId,
        photoUrl: `${req.protocol}://${req.get("host")}/uploads/webp/${student.photoWebP}`,
      },
    });
  } catch (error) {
    console.error("Error:", error);
    res.status(500).json({ error: "Failed to create student" });
  }
};

// Get Students
exports.listStudents = async (req, res) => {
  try {
    const { page = 1, limit = 10, teacherId, search = "" } = req.query;
    const offset = (parseInt(page) - 1) * parseInt(limit);

    const where = {};
    if (teacherId) where.teacherId = teacherId;
    if (search) where.search = search;

    const { rows: students, count } = Student.findAll({
      where,
      limit: parseInt(limit),
      offset,
    });

    const studentsWithUrls = students.map((student) => ({
      id: student.id,
      name: student.name,
      age: student.age,
      phone: student.phone,
      fatherName: student.fatherName,
      description: student.description,
      teacherId: student.teacherId,
      photoUrl: `${req.protocol}://${req.get("host")}/uploads/webp/${student.photoWebP}`,
      createdAt: student.created_at,
    }));

    res.json({
      data: studentsWithUrls,
      total: count,
      page: parseInt(page),
      limit: parseInt(limit),
    });
  } catch (error) {
    console.error("Error:", error);
    res.status(500).json({ error: "Failed to load students" });
  }
};

// Get Single Student
exports.getStudentById = async (req, res) => {
  try {
    const student = await Student.findById(req.params.id);
    if (!student) {
      return res.status(404).json({ error: "Student not found" });
    }

    res.json({
      id: student.id,
      name: student.name,
      phone: student.phone,
      fatherName: student.fatherName,
      description: student.description,
      teacherId: student.teacher_id,
      photoUrl: `${req.protocol}://${req.get("host")}/uploads/webp/${student.photoWebP}`,
      createdAt: student.created_at,
    });
  } catch (error) {
    console.error("Error:", error);
    res.status(500).json({ error: "Failed to load student" });
  }
};

// Update Student
exports.updateStudent = async (req, res) => {
  try {
    const student = await Student.findById(req.params.id);
    if (!student) {
      return res.status(404).json({ error: "Student not found" });
    }

    const { name, age, phone, fatherName, description, teacherId } = req.body;

    const updateData = {
      name: name ?? student.name,
      age: age !== undefined ? parseInt(age) : student.age,
      phone: phone ?? student.phone,
      fatherName: fatherName ?? student.fatherName,
      description: description ?? student.description,
      teacherId: teacherId ?? student.teacher_id,
    };

    // If new photo uploaded
    if (req.file) {
      // Delete old image
      if (student.photo_webp) {
        const oldPath = path.join(
          __dirname,
          "..",
          "uploads",
          "webp",
          student.photo_webp,
        );
        if (fs.existsSync(oldPath)) {
          fs.unlinkSync(oldPath);
        }
      }

      // Process new image
      const timestamp = Date.now();
      const cleanFilename = `student-${timestamp}.webp`;
      const webpPath = path.join(
        __dirname,
        "..",
        "uploads",
        "webp",
        cleanFilename,
      );

      await sharp(req.file.path)
        .resize(500, 500)
        .webp({ quality: 80 })
        .toFile(webpPath);

      updateData.photo = req.file.originalname;
      updateData.photoWebP = cleanFilename;

      // Delete temp uploaded file
      fs.unlinkSync(req.file.path);
    }

    const updatedStudent = Student.update(req.params.id, updateData);

    res.json({
      message: "Student updated",
      student: {
        id: updatedStudent.id,
        name: updatedStudent.name,
        age: updatedStudent.age,
        phone: updatedStudent.phone,
        fatherName: updatedStudent.fatherName,
        description: updatedStudent.description,
        teacherId: updatedStudent.teacherId,
        photoUrl: updatedStudent.photo
          ? `${req.protocol}://${req.get("host")}/uploads/webp/${updatedStudent.photoWebP}`
          : null,
        createdAt: updatedStudent.created_at,
        updatedAt: updatedStudent.updated_at,
      },
    });
  } catch (error) {
    console.error("Error:", error);
    res.status(500).json({ error: "Failed to update student" });
  }
};

// Delete Student
exports.deleteStudent = async (req, res) => {
  try {
    const student = await Student.findById(req.params.id);
    if (!student) {
      return res.status(404).json({ error: "Student not found" });
    }

    // Delete image file
    if (student.photo_webp) {
      const imagePath = path.join(
        __dirname,
        "..",
        "uploads",
        "webp",
        student.photo_webp,
      );
      if (fs.existsSync(imagePath)) {
        fs.unlinkSync(imagePath);
      }
    }

    Student.delete(req.params.id);

    res.json({ message: "Student deleted" });
  } catch (error) {
    console.error("Error:", error);
    res.status(500).json({ error: "Failed to delete student" });
  }
};

// Get students by teacher
exports.getStudentsByTeacher = async (req, res) => {
  try {
    const { teacherId } = req.params;
    const { page = 1, limit = 10 } = req.query;
    const offset = (parseInt(page) - 1) * parseInt(limit);

    const { rows: students, count } = Student.findByTeacher(teacherId, {
      limit: parseInt(limit),
      offset,
    });

    const studentsWithUrls = students.map((student) => ({
      id: student.id,
      name: student.name,
      age: student.age,
      phone: student.phone,
      fatherName: student.fatherName,
      description: student.description,
      teacherId: student.teacherId,
      photoUrl: `${req.protocol}://${req.get("host")}/uploads/webp/${student.photoWebP}`,
      createdAt: student.created_at,
    }));

    res.json({
      data: studentsWithUrls,
      total: count,
      page: parseInt(page),
      limit: parseInt(limit),
      teacherId,
    });
  } catch (error) {
    console.error("Error:", error);
    res.status(500).json({ error: "Failed to load students" });
  }
};
